/*
 * Based on systemd's macro.h
 * https://www.freedesktop.org/wiki/Software/systemd/
 * Modified by Qualys, January 2019
 */

/* SPDX-License-Identifier: GPL-3.0-or-later */
#pragma once

#include <limits.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define die() do { \
    printf("died in %s:%u, %s().\n", __FILE__, __LINE__, __func__); \
    exit(EXIT_FAILURE); \
} while (0)

#define xwrite(fd, buf, count) do { \
    const size_t _count_ = (count); \
    if (_count_ <= 0) die(); \
    if (_count_ > SSIZE_MAX) die(); \
    if (write((fd), (buf), _count_) != (ssize_t)_count_) die(); \
} while (0)

#define xsnprintf(str, size, format, ...) ({ \
    const size_t _size_ = (size); \
    if (_size_ <= 0) die(); \
    if (_size_ >= INT_MAX) die(); \
    const unsigned _len_ = snprintf((str), _size_, (format), __VA_ARGS__); \
    if (_len_ <= 0) die(); \
    if (_len_ >= _size_) die(); \
    _len_; \
})

#define _packed_ __attribute__((__packed__))

static inline size_t ALIGN_TO(size_t l, size_t ali) {
        if (ali < 1) die();
        if (l > SIZE_MAX - ali) die();
        return ((l + ali - 1) & ~(ali - 1));
}

#ifndef __COVERITY__
#  define VOID_0 ((void)0)
#else
#  define VOID_0 ((void*)0)
#endif

#define ELEMENTSOF(x)                                                   \
        (__builtin_choose_expr(                                         \
                !__builtin_types_compatible_p(typeof(x), typeof(&*(x))), \
                sizeof(x)/sizeof((x)[0]),                               \
                VOID_0))

