#!/usr/bin/perl -w
use IO::File;
use IO::Handle;
# use Fcntl ':flock';
use strict;
use SerialPort 0.03;

# Adjust this value to timeout a connection attempt sooner (in seconds)
my $dial_timeout=60;

my %modem_responses=('OK' => 'OK', 
                     'ERROR' => 'ERROR',
                     'DIALTONE' => 'NO DIALTONE',
                     'CARRIER' => 'NO CARRIER',
                     'BUSY' => 'BUSY',
                     'RINGING' => 'RINGING',
# This next one should positively identify a FAX machine - Check this
                     'FAX' => '+FNSF', 
                     'NO FAX CARRIER' => '+FHNG:11', 
                     'CONNECT' => 'CONNECT');

$main::binverbose=1; # prints binary dump trace to STDERR

my %results;
my $sighup=0;
$SIG{HUP}=sub {print "Caught HUP\n"; $sighup++;};

my $modem=open_modem("/dev/modem");
reset_modem($modem);

while (<>) {
  my $number=$_;
  chomp $number;
  open_logfiles($number);
  
  check_data($modem,$number);
  reset_modem($modem);
  # Some fax machines show up as DATA
  if ($results{$number}=~/DATA-CONNECT/) {
    check_fax($modem,$number);
    reset_modem($modem);
  }

  &close_logfiles;
}

#flock($modem->{FH},LOCK_UN) or die "Couldn't unlock the modem: $!";
$modem->close;
undef $modem;

my $number;
foreach $number (sort keys %results) {
  print "$number: $results{$number}\n";
}

exit;

# this routine is used to record just what we send to and receive from
# the serial port.
# We can log to two files an ASCII dump, and a HEX/ASCII dump
# the HEX/ASCII dump shows up to 15 characters per line in hex, with
# the ascii representation in the last 15 characters of the line.
# Ideally, I should consolidate partial lines in the same direction
# into one line, but I think that that can be post-processed
sub trace {
  my $direction=shift;
  my $string=shift;
  my $i;
  my $outstring;
  my $charno;
  my $char;
  
  if (defined $main::ascdump) {printf $main::ascdump "$string";}
  if ($direction=~/Read/i) { $direction="< "; }
  elsif ($direction=~/Write/i) { $direction="> "; }
  for ($i=0; $i<length($string); $i++) {
    $charno=$i%15;
    if ($charno == 0) {
      if ($i>0) { 
        if (defined $main::bindump) {print $main::bindump "$outstring\n"; }
        if ($main::binverbose) {print STDERR "$outstring\n";}
      }
      $outstring=$direction.' 'x78;
    }
    $char=ord(substr($string,$i,1));
    substr($outstring,3+$charno*3,3)=sprintf("%02x ",$char);
    if (($char < 32) or ($char > 126)) { $char=32; }
    substr($outstring,55+$charno,1)=chr($char);
  }
  if ($charno>0) { 
    if (defined $main::bindump) {print $main::bindump "$outstring\n";}
    if ($main::binverbose) {print STDERR "$outstring\n";}
  }
}

# This subroutine checks to see if the reply that we have received from the
# modem is one of the known MODEM responses
sub is_modem_reply {
  my $string=shift;
  my $i;

  foreach $i (keys %modem_responses) {
    if ($string =~ /\Q$modem_responses{$i}/) {return $i;}
  }
  return 0;
}

sub read_modem {
  my $modem=shift;
  my $timeout=shift;
  my $return_on_modem=shift;
  my $rin=''; my $rout='';
  my $response='';

  # if $return_on_modem is not set to non-zero, we read until the timeout
  if (!defined $return_on_modem) {$return_on_modem=0;}

  if (!defined $timeout) { die "Must specify a timeout for read_modem\n";}
  if ($timeout<=0) {die "Can't have a non-positive timeout\n";}

  # do the loop until we get a reply from the modem or $timeout occurs
  eval {
    local $SIG{ALRM} = sub { die "alarm\n" };       # NB \n required
    alarm $timeout;
    do {
      my $chars=0;
      my $charsread='';
      ($chars,$charsread)=$modem->read(1000);
          if (!defined $chars || ($chars == 0)) { 
        select (undef,undef,undef,0.25); 
      } else {
        $response=$response.$charsread;
        trace("Read:",$charsread); 
      }
      if ($return_on_modem and is_modem_reply($response)) { 
        alarm 0;
        return; 
      }
      if ($sighup) {
        $sighup=0;
        alarm 0;
        return;
      }
    } while (1);
  };
  die if $@ && $@ ne "alarm\n";
#  if ($@ eq "alarm\n") { 
#    print STDERR "read_modem: Timed out after $timeout seconds\n";
#  }
  return $response;
}

sub write_modem {
  my $modem=shift;
  my $command=shift;
  my $chars=0;

  $chars = $modem->write($command);
  warn "write failed\n"         unless ($chars);
  warn "write incomplete\n"     if ( $chars != length($command) );
  trace("Write:",$command);
}

sub modem_command {
  my $modem=shift;
  my $command=shift;
  my $modem_timeout=shift;
  my $response='';

  if (!($command =~ /\r/)) { warn "No CR in command '$command'\n";}
  write_modem($modem,$command);
  
  # Wait for the response from the modem
  # We are waiting for the modem to give us a recognised response
  #  e.g. OK, No Dialtone, busy, ERROR, etc
  #  set the third parameter to non-zero to terminate the read when
  # the modem has responded
  $response=read_modem($modem,$modem_timeout,1);
  return (is_modem_reply($response),$response);
}

sub open_modem {
  my $device=shift;
  my $lockfile=$device; 

  $modem = SerialPort->new ($device) || die;
  $modem->baudrate(9600)     || die "fail setting baudrate";
  $modem->parity("none")     || die "fail setting parity";
  $modem->databits(8)        || die "fail setting databits";
  $modem->stopbits(1)        || die "fail setting stopbits";
  $modem->handshake("rts")   || die "fail setting handshake";

  $lockfile =~ s/\/.*\///;
  $lockfile="/var/lock/LCK.." . $lockfile;
  if (-f $lockfile ) {
    die "$device is locked: $lockfile exists!\n";
  }
  if (!defined $modem) { die "Couldn't open $device : $!"; }
#  flock($modem->{FH},LOCK_EX|LOCK_NB) or die "Couldn't lock the modem: $!";
  return $modem;
}

sub open_logfiles {
  my $number=shift;

  if (defined $number) {
    $main::bindump=new IO::File "$number.bin", "a";
    if (!defined $main::bindump) { die "Couldn't open binary trace file : $!"; }
    $main::bindump->autoflush(1);
    $main::ascdump=new IO::File "$number.asc", "a";
    if (!defined $main::ascdump) { die "Couldn't open ASCII trace file : $!"; }
    $main::ascdump->autoflush(1);
  } else {
    print STDERR "No number given. Not logging!\n"; }
}

sub close_logfiles {
  if (defined $main::bindump) { 
    $main::bindump->close || die "Error closing $main::bindump\n";
  }
  if (defined $main::ascdump) { 
    $main::ascdump->close || die "Error closing $main::ascdump\n";
  }
}

sub check_data {
  my $modem=shift;
  my $number=shift;
  my $resp; my $in;

  # Check for a DATA connection
  # We can only dial if DTR is high - otherwise we just get OK immediately
  $modem->dtr_active(1);
  ($resp,$in)=modem_command($modem,"ATDT$number\r\n",$dial_timeout);
  if ($resp eq 'DIALTONE') {
    die "No dialtone - please fix the problem and try again\n";
  } elsif ($resp eq 'BUSY') {
    print "$number: BUSY\n";
    $results{$number}.='DATA-BUSY/';
  } elsif ($resp eq 'CARRIER') {
    print "$number: NO CARRIER\n";
    $results{$number}.='DATA-NO CARRIER/';
  } elsif ($resp eq 'CONNECT') {
    my $pos=index($in,$modem_responses{$resp})+length($modem_responses{$resp});
    $in=substr($in,$pos);
    print "$number: DATA-CONNECT\n";
    $results{$number}.='DATA-CONNECT/';
    # See what we get without prodding
    $in=read_modem($modem,5);
    # We could probably put some more fancy algorithms in here to do automatic
    # system detection. e.g. PC-Anywhere, Windows RAS, Shiva, etc
    # and try some password guessing perhaps
    # Prod the other end with a couple of <CR>
    write_modem($modem,"\r\n\r\n");
    $in=read_modem($modem,5);
    # Do it again
    write_modem($modem,"\r\n\r\n");
    $in=read_modem($modem,5);
  } else {
    print "$number: Unknown response. Received ($in) resp=$resp\n";
    $results{$number}.='DATA-Unknown response/';
  }
  # This should hang up the modem - we do this in reset_modem too,
  # but this way DTR is the same at exit as it is at entry
  # no side effects to this routine :-)
  $modem->dtr_active(0);
}

sub check_fax {
  my $modem=shift;
  my $number=shift;
  my $resp; my $in;

  # Let's check if this is a fax machine
  # How does one positively identify a fax machine? HELP-FIXME
  # This could be erroneous - what if the machine is auto detecting FAX/DATA?
  $modem->dtr_active(1);
  ($resp,$in)=modem_command($modem,"AT+FCLASS=2\r\n",2);
  if ($resp ne 'OK') {
    die "Expected OK, received ($in) resp=$resp\n";
  }
  ($resp,$in)=modem_command($modem,"ATDT$number\r\n",$dial_timeout);
  if ($resp eq 'DIALTONE') {
    die "No dialtone - please fix the problem and try again\n";
  } elsif ($resp eq 'BUSY') {
    print "$number: FAX-BUSY\n";
    $results{$number}.='FAX-BUSY/';
  } elsif ($resp eq 'CARRIER') {
    print "$number: FAX-NO CARRIER\n";
    $results{$number}.='FAX-NO CARRIER/';
  } elsif ($resp eq 'NO FAX CARRIER') {
    print "$number: FAX-NO FAX CARRIER\n";
    $results{$number}.='FAX-NO FAX CARRIER/';
  } elsif ($resp eq 'FAX') {
    print "$number: FAX\n";
    $results{$number}.='FAX-CONNECT/';
    #let's get the rest of the banner
    $in=read_modem($modem,5);
  } else {
    print "$number: Unknown response. Received ($in) resp=$resp\n";
    $results{$number}.='FAX-Unknown response/';
  }
  $modem->dtr_active(0);
}

sub reset_modem {
  my $modem=shift;
  my $in;
  my $resp;

  # Hang up cleanly under most all circumstances
  $modem->dtr_active(0);
  # Loop while Carrier Detect is set
#  while ($modem->dtr_active) {
#    select(undef,undef,undef,0.25);
#  }
  # Clear the serial buffer
  $in=read_modem($modem,2,1);
  if (length($in) > 0) { print "Received ($in) after hanging up\n"; }
  # sleep a bit
  select(undef,undef,undef,1);
  # reset the modem to default state - we can probably optimise this
  ($resp,$in)=modem_command($modem,"ATZ\r\n",2);
  if ($resp ne 'OK') {
    die "Expected OK, received ($in) resp=$resp\n";
  }
  ($resp,$in)=modem_command($modem,"ATV1E0\r\n",2);
  if ($resp ne 'OK') {
    die "Expected OK, received ($in) resp=$resp\n";
  }
}


