/************************************************************************
*      +--------------------------------------------------------+       *
*      | Proyecto de Fin de Carrera de Fernando Migulez Palomo |       *
*      |          Dpto. Electrnica y Comunicaciones            |       * 
*      |          Grupo de Ingeniera Telemtica (GIT)          |       *
*      |            E.T.S.I.I. y I.T. Bilbao (Espaa)           |       *
*      +--------------------------------------------------------+       *
*                                                                       *
*  Filename: pinj.c                                                     *
*  Author: Fernando Migulez Palomo                                     *
*  Version 1.0                                                          *
*  Description: Packet injector program. It re-injects traffic from a   *
*               tcpdump file to the specified network interface         * 
*                                                                       *
*  Changes: Oct-3-2001  Fernando Migulez starts coding the module      *
*                                                                       *
*************************************************************************/

/*  I N C L U D E S  **********************************************************/
#include <pcap.h>
#include <sys/socket.h>
#include <linux/if_ether.h>
#include <unistd.h>
#include <sys/timeb.h>

/*  D E F I N E S *************************************************************/
#define DEVICE "eth1"
#define STEP_INFO 100

/*  D E F I N T I O N S *******************************************************/
void heading(void)
{    
    printf("PINJ: Packet INJector. (c) Fernando Migulez Palomo\n");
    printf("---------------------------------------------------\n");
}

void usage(char *progname)
{
    fprintf(stderr, "Usage: %s [-i if] [-t time] [-q] log_filename\n", progname);
    fprintf(stderr, "       log_filename   network traffic file in tcpdump format\n");
    fprintf(stderr, "       if             network interface name. By default '%s'\n", DEVICE);
    fprintf(stderr, "       time           useconds to wait between consecutive packet injections\n");
    fprintf(stderr, "       q              quiet mode (slightly faster)\n");
}

/*  M A I N *******************************************************************/
int main (int argc, char *argv[])
{
    pcap_t * pd;
    char error_buf_pcap[256];
    char error_buf_net[256];
    u_char *pkt;
    struct pcap_pkthdr pkt_hdr;
    struct libnet_link_int * network;
    int i, c;
    unsigned long idle_us;
    char device[10];
    int bytes, pkt_len, quiet;
    struct timeb init_time;
    struct timeb final_time;
   
    /* Default values */
    strncpy(device, DEVICE, 5);
    idle_us = 0;
    quiet = 0;
    
    heading();

    while ((c = getopt(argc, argv, "i:t:q")) != EOF)
    {
        switch (c)
        {
            case 'i':
                strncpy(device, optarg, 10);
                break;  
            case 't':
                idle_us = atol(optarg);
                break;
            case 'q':
                quiet = 1;
                break;
            default:
                usage(argv[0]);
                exit(1);
        }
    }

    if (optind == argc)
    {
        usage(argv[0]);
        exit(1);
    }

    if ((pd = pcap_open_offline(argv[optind], error_buf_pcap)) == NULL)
    {
        fprintf(stderr, "Error opening '%s' file: %s\n",argv[optind],error_buf_pcap);
        exit(1);
    }

    if ((network = libnet_open_link_interface(device, error_buf_net)) == NULL)
    {
        fprintf(stderr, "Error opening raw socket: %s\n", error_buf_net);
        pcap_close(pd);
        exit(1);
    }

    if (idle_us != 0)
        printf("Idle time between packet injections: %d usec\n", idle_us);

    printf("Sending on interface: %s\n", device);
    printf("Traffic file: %s\n", argv[optind]);

    ftime(&init_time);
    bytes = 0;

    i = 1;
    while ((pkt = pcap_next(pd, &pkt_hdr)) != NULL)
    {
        if ((pkt_len = libnet_write_link_layer(network, device, pkt, pkt_hdr.caplen)) <= 0)
        {
            fprintf(stderr, "Unable to write to network: %s\n", error_buf_net);
            break;
        }
      
        bytes += pkt_len;
 
        if (idle_us)
            usleep(idle_us);

        if (i % STEP_INFO == 0 && !quiet)
            printf("Packet %d written to wire\n", i);
       
        i++;
    }
    
    ftime(&final_time);
    printf("Total number of injected packets: %d\n", i-1);
    printf("Total amount of injected data: %f KB\n", bytes/1024.0);
    printf("Average injection speed: %f KB/s - %f pkts/s\n", 
           bytes/( (final_time.time*1000.0 + final_time.millitm - 
                   init_time.time*1000.0 - init_time.millitm) * 1.024),
           (i-1)/( (final_time.time*1000.0 + final_time.millitm - 
                   init_time.time*1000.0 - init_time.millitm) * 1.024));
    if (strlen(pcap_geterr(pd)) != 0)
        printf("Pcap error: %s\n", pcap_geterr(pd));
 
    pcap_close(pd);
    libnet_close_link_interface(network);

    return 0;
}
